#include <iostream>
#include <cmath>

using std::string;

#include "SDL/SDL.h"

#include "options.h"
#include "painter.h"

//constructor

Painter::Painter(Options options_):
  screenBPP(32),
  options(options_),
  screen(NULL),
  tileSize(14),
  flags(0),
  current_face(":)")
{
  SDL_Surface *icon = SDL_LoadBMP("ms3.bmp");
  SDL_WM_SetIcon(icon, NULL);
  screen = SDL_SetVideoMode(
    getWindowWidth(), 
    getWindowHeight(),
    screenBPP,
    SDL_SWSURFACE);
  if (options.columns > 13)
    SDL_WM_SetCaption("Minesweeper 3", NULL);
  else
    SDL_WM_SetCaption("MS3", NULL);
  tileset = SDL_LoadBMP("tileset.bmp");
  tileset = SDL_DisplayFormat(tileset);
}

//destructor

Painter::~Painter()
{
  SDL_FreeSurface(screen);
  SDL_FreeSurface(tileset);
}

//interface

void Painter::drawGrid()
{
  //grid
  SDL_Rect srcrec, dstrec;
  dstrec.w = tileSize;
  dstrec.h = tileSize;
  srcrec.w = tileSize;
  srcrec.h = tileSize;
  for (int i = 0; i < options.rows; i++)
  {
    for (int j = 0; j < options.columns; j++)
    {
      dstrec.x = j * tileSize;
      dstrec.y = (i + 1) * tileSize;
      srcrec.x = 0;
      srcrec.y = 14;
      SDL_BlitSurface(tileset, &srcrec, screen, &dstrec);
    }
  }
  //newgame icon and time
  srcrec.x = 42;
  srcrec.y = 42;
  dstrec.x = 3 * tileSize;
  dstrec.y = (options.rows + 1) * tileSize;
  SDL_BlitSurface(tileset, &srcrec, screen, &dstrec);
  writeTime(0);
  //flags icon and flag count
  srcrec.x = 14;
  srcrec.y = 42;
  dstrec.x = (options.columns - 4) * tileSize;
  dstrec.y = (options.rows + 1) * tileSize;
  SDL_BlitSurface(tileset, &srcrec, screen, &dstrec);
  writeFlags(0);
  //flip screen
  SDL_Flip(screen);
}

void Painter::drawDigitAt(int x, int y, int digit)
{
  SDL_Rect srcrec = {digit * tileSize, 0, tileSize, tileSize};
  SDL_Rect dstrec = {x * tileSize, (y + 1) * tileSize, tileSize, tileSize};
  SDL_BlitSurface(tileset, &srcrec, screen, &dstrec);
  SDL_Flip(screen);
}

void Painter::drawFlagAt(int x, int y)
{
  SDL_Rect srcrec = {0, 42, tileSize, tileSize};
  SDL_Rect dstrec = {x * tileSize, (y + 1) * tileSize, tileSize, tileSize};
  SDL_BlitSurface(tileset, &srcrec, screen, &dstrec);
  SDL_Flip(screen);
}

void Painter::drawHiddenTileAt(int x, int y)
{
  drawHiddenTileAt(x, y, 0);
}

void Painter::drawHiddenTileAt(int x, int y, int phase)
{
  SDL_Rect srcrec = {phase * 14, 14, tileSize, tileSize};
  SDL_Rect dstrec = {x * tileSize, (y + 1) * tileSize, tileSize, tileSize};
  SDL_BlitSurface(tileset, &srcrec, screen, &dstrec);
  SDL_Flip(screen);
}

void Painter::showMine(int x, int y)
{
  SDL_Rect srcrec = {0, 28, tileSize, tileSize};
  SDL_Rect dstrec = {x * tileSize, (y + 1) * tileSize, tileSize, tileSize};
  SDL_BlitSurface(tileset, &srcrec, screen, &dstrec);
  SDL_Flip(screen);
}

void Painter::showKillerMine(int x, int y)
{
  SDL_Rect srcrec = {14, 28, tileSize, tileSize};
  SDL_Rect dstrec = {x * tileSize, (y + 1) * tileSize, tileSize, tileSize};
  SDL_BlitSurface(tileset, &srcrec, screen, &dstrec);
  SDL_Flip(screen);
}

void Painter::showFalseMine(int x, int y)
{
  SDL_Rect srcrec = {28, 28, tileSize, tileSize};
  SDL_Rect dstrec = {x * tileSize, (y + 1) * tileSize, tileSize, tileSize};
  SDL_BlitSurface(tileset, &srcrec, screen, &dstrec);
  SDL_Flip(screen);
}

void Painter::showFace(int phase)
{
  showFace(current_face, phase);
}

void Painter::showFace(std::string face, int phase)
{
  
  SDL_Rect srcrec = {-1, -1, tileSize, tileSize};
  current_face = face;
  if (face == ":o")
  {
    srcrec.x = 42 + (phase * tileSize);
    srcrec.y = 28;
  }
  else if (face == ":)")
  {
    srcrec.x = 42 + (phase * tileSize);
    srcrec.y = 42;
  }
  else if (face == "8D")
  {
    srcrec.x = 98 + (phase * tileSize);
    srcrec.y = 42;
  }
  else if (face == ":(")
  {
    srcrec.x = 98 + (phase * tileSize);
    srcrec.y = 28;
  }
  SDL_Rect dstrec = {3 * tileSize, (options.rows + 1) * tileSize, tileSize, tileSize};
  SDL_BlitSurface(tileset, &srcrec, screen, &dstrec);
  SDL_Flip(screen);  
}

void Painter::writeFlags(int number)
{
  writeBlackNumber(number, options.columns - 3);
}

void Painter::writeTime(int number)
{
  writeBlackNumber(number, 0);
}

//internal functions

int Painter::getWindowWidth()
{
  return options.columns * tileSize;
}

int Painter::getWindowHeight()
{
  return (options.rows + 2) * tileSize;
}

int Painter::writeBlackNumber(int number, int x)
{
  //breaking down numbers
  if (number > 999)
    number = 999;
  int hundreds = static_cast<int> ( floor(number / 100) );
  int tens = static_cast<int> ( floor( ( number - (hundreds * 100) ) /10 ) );
  int units = number - (hundreds * 100) - (tens * 10);
  //blitting
  //SDL_Rect = {x, y, w, h}
  SDL_Rect srcrec = {hundreds * tileSize, 56, tileSize, tileSize};
  SDL_Rect dstrec = {x * tileSize, (options.rows + 1) * tileSize, tileSize, tileSize};
  SDL_BlitSurface(tileset, &srcrec, screen, &dstrec);
  srcrec.x = tens * tileSize;
  dstrec.x += tileSize;
  SDL_BlitSurface(tileset, &srcrec, screen, &dstrec);
  srcrec.x = units * tileSize;
  dstrec.x += tileSize;
  SDL_BlitSurface(tileset, &srcrec, screen, &dstrec);
  //flip screen
  SDL_Flip(screen);
}
