#include <iostream>
#include <vector>
#include <cmath>

#include "SDL/SDL.h"
#include "SDL/SDL_image.h"

#include "options.h"
#include "window.h"
#include "tile.h"
#include "game.h"

Game::Game(Options options):
  gameoptions(options),
  gameover(false),
  started(false),
  flags(0),
  time_started(0),
  won(false)
{
  hover_tile.x = 0;
  hover_tile.y = 0;
  gamewindow = new Window(options);
  initImages();
  initGrid();
}

Game::~Game()
{
  delete gamewindow;
}

void Game::initImages()
{
  SDL_Surface *tempimg = IMG_Load("msfont1.bmp");
  font1 = SDL_DisplayFormat(tempimg);
}

void Game::initGrid()
{
  grid.resize(gameoptions.columns);
  for (int i=0; i<gameoptions.columns; i++) {
    grid[i].resize(gameoptions.rows);
    for (int j=0; j<grid[i].size(); j++) {
      grid[i][j].hidden = true;
      grid[i][j].mined = false;
      grid[i][j].flagged = false;
      gamewindow->drawTile(font1, 'H', i, j);
    }
  }
}

int random(int min, int max)
{
  //generate random number and put it in desired range
  int range = (max+1) - min;
  int random_number = rand()%range;
  random_number += min;
  //return random number
  return random_number;
}

void Game::generateMines(unsigned short int x, unsigned short int y)
{
  srand((unsigned)time(NULL));
  int rand_x, rand_y;
  for (int i=0; i<gameoptions.mines; i++) {
    int rand_x = random(0, gameoptions.columns-1);
    int rand_y = random(0, gameoptions.rows-1);
    if (grid[rand_x][rand_y].mined == true || (x == rand_x && y == rand_y))
      i--;
    else {
      grid[rand_x][rand_y].mined = true;
      if (gameoptions.cheat == true)
        gamewindow->drawTile(font1, 'M', rand_x, rand_y);
    }
  }
}

void Game::updateHoverTile(unsigned short int x, unsigned short int y,
        bool leftmousebuttondown)
{
  //determine what hover tile to show, depending on state of left mouse button
  char hovertilechar;
  if (leftmousebuttondown == true)
    hovertilechar = '0';
  else
    hovertilechar = 'I';
  //update the hover tile
  if (y < gameoptions.rows)
  {
    if (grid[x][y].hidden == true &&
            grid[x][y].flagged == false &&
            gameover == false)
    {
      if (x != hover_tile.x || y != hover_tile.y ||
              leftmousebuttondown == true)
      {
        if (grid[hover_tile.x][hover_tile.y].hidden == true) {
          if (grid[hover_tile.x][hover_tile.y].flagged == false) {
            if (grid[hover_tile.x][hover_tile.y].mined == true &&
                  gameoptions.cheat == true)
              gamewindow->drawTile(font1, 'M', hover_tile.x, hover_tile.y);
            else
              gamewindow->drawTile(font1, 'H', hover_tile.x, hover_tile.y);
          }
          else
            gamewindow->drawTile(font1, 'F', hover_tile.x, hover_tile.y);
        }
        hover_tile.x = x;
        hover_tile.y = y;
        gamewindow->drawTile(font1, hovertilechar, hover_tile.x, hover_tile.y);
      }
    }
  }
}

void Game::clearTileAt(unsigned short int x, unsigned short int y)
{
  if (y < gameoptions.rows)
  {
    if (grid[x][y].hidden == true &&
            grid[x][y].flagged == false &&
            gameover == false)
    {
      grid[x][y].hidden = false;
      if (grid[x][y].mined == true)
        loseGame(x, y);
      else
      {
        gamewindow->drawTile(font1, '0', x, y);
        if (countAdjacentMinesAt(x, y) == 0)
          clearAdjacentTilesAt(x, y);
        else
          showAdjacentMinesAt(x, y);
      }
    }
    //check if player won
    unsigned short int hiddentiles = 0;
    for (int i=0; i<gameoptions.columns; i++)
    {
      for (int j=0; j<gameoptions.rows; j++)
      {
        if (grid[i][j].hidden == true)
          hiddentiles++;
      }
    }
    if (hiddentiles == gameoptions.mines)
      winGame();
  }
}

void Game::setFlagAt(unsigned short int x, unsigned short int y)
{
  if (grid[x][y].hidden == true && gameover == false)
  {
    if (grid[x][y].flagged == false)
    { //tile is not flagged => flag it
      grid[x][y].flagged = true;
      gamewindow->drawTile(font1, 'F', x, y);
      flags++;
      gamewindow->writeInBlack(gameoptions.columns-3, flags);
    }
    else
    { //tile is already flagged => unflag it
      grid[x][y].flagged = false;
      gamewindow->drawTile(font1, 'H', x, y);
      flags--;
      gamewindow->writeInBlack(gameoptions.columns-3, flags);
    }
  }
}

const unsigned short int Game::countAdjacentMinesAt(unsigned short int x,
        unsigned short int y)
{
  unsigned short int minesnear = 0;
  short int xstart = -1;
  short int xend = 1;
  short int ystart = -1;
  short int yend = 1;
  if ((x+1) == gameoptions.columns)
    xend = 0;
  if (x == 0)
    xstart = 0;
  if ((y+1) == gameoptions.rows)
    yend = 0;
  if (y == 0)
    ystart = 0;
  for (int i = xstart; i <= xend; i++) {
    for (int j = ystart; j <= yend; j++) {
      if (grid[x+i][y+j].mined == true)
        minesnear++;
    }
  }
  return minesnear;
}

const unsigned short int Game::showAdjacentMinesAt(unsigned short int x,
        unsigned short int y)
{
  unsigned short int minesnear = countAdjacentMinesAt(x,y);
  const char showfigure = static_cast<char>(minesnear+48);
  gamewindow->drawTile(font1, showfigure, x, y);
}

void Game::clearAdjacentTilesAt(unsigned short int x, unsigned short int y)
{
  unsigned short int minesnear = 0;
  short int xstart = -1;
  short int xend = 1;
  short int ystart = -1;
  short int yend = 1;
  if ((x+1) == gameoptions.columns)
    xend = 0;
  if (x == 0)
    xstart = 0;
  if ((y+1) == gameoptions.rows)
    yend = 0;
  if (y == 0)
    ystart = 0;
  for (int i = xstart; i <= xend; i++) {
    for (int j = ystart; j <= yend; j++)
      clearTileAt(x + i, y + j);
  }
}

void Game::loseGame(unsigned short int x, unsigned short int y)
{
  //set gameover flag
  gameover = true;
  //show all mines and check flags
  for (int i=0; i<gameoptions.columns; i++)
  {
    for (int j=0; j<gameoptions.rows; j++)
    {
      if (grid[i][j].flagged == true)
      {
        if (grid[i][j].mined != true)
          gamewindow->drawTile(font1, 'X', i, j);
      }
      else if (grid[i][j].mined == true)
        gamewindow->drawTile(font1, 'M', i, j);
    }
  }
  //show killer mine
  gamewindow->drawTile(font1, 'K', x, y);
  gamewindow->writeInBlack(3, "T");
}

void Game::winGame()
{
  gameover = true;
  gamewindow->writeInBlack(3, "W");
  won = true;
}

void Game::handleDoubleClick(unsigned short int x, unsigned short int y)
{
  if (countAdjacentFlagsAt(x, y) == countAdjacentMinesAt(x, y)
    && grid[x][y].hidden == false)
  {
    short int xstart = -1;
    short int xend = 1;
    short int ystart = -1;
    short int yend = 1;
    if ((x+1) == gameoptions.columns)
      xend = 0;
    if (x == 0)
      xstart = 0;
    if ((y+1) == gameoptions.rows)
      yend = 0;
    if (y == 0)
      ystart = 0;
    for (int i = xstart; i <= xend; i++) {
      for (int j = ystart; j <= yend; j++)
        clearTileAt(x + i, y + j);
    }
  }
}

const unsigned short int Game::countAdjacentFlagsAt(unsigned short int x,
        unsigned short int y)
{
  unsigned short int flagsnear = 0;
  short int xstart = -1;
  short int xend = 1;
  short int ystart = -1;
  short int yend = 1;
  if ((x+1) == gameoptions.columns)
    xend = 0;
  if (x == 0)
    xstart = 0;
  if ((y+1) == gameoptions.rows)
    yend = 0;
  if (y == 0)
    ystart = 0;
  for (int i = xstart; i <= xend; i++) {
    for (int j = ystart; j <= yend; j++) {
      if (grid[x+i][y+j].flagged == true)
        flagsnear++;
    }
  }
  return flagsnear;
}

void Game::showCurrentTime()
{
  const unsigned int current_time = static_cast<unsigned int>
          (floor( (SDL_GetTicks() - time_started) / 1000 ));
  if (started == true && gameover == false)
    gamewindow->writeInBlack(0, current_time);
}

bool Game::gameIsOver()
{
  return gameover;
}
